
 /**
  * @package subvert
  *
  * @file Main
  * @copyright (c) 2011 Christoph Kappel <unexist@dorfelite.net>
  * @version $Id: src/shared/shared.c,v 2552 2011/01/16 20:25:07 unexist $
  *
  * This program can be distributed under the terms of the GNU GPL.
  * See the file COPYING.
  **/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <signal.h>
#include <execinfo.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>

/* Globals {{{ */
static Display *display = NULL;
static Window victim = None, culprit = None;
static int running = True;
/* }}} */

/* SubvertError {{{ */
static int
SubvertXError(Display *disp,
  XErrorEvent *ev)
{
  char error[255] = { 0 };

  XGetErrorText(disp, ev->error_code, error, sizeof(error));
  printf(">>> %s: win=%#lx, request=%d\n",
    error, ev->resourceid, ev->request_code);

  return 0;
} /* }}} */

/* SubtleSignal {{{ */
static void
SubvertSignal(int signum)
{
  switch(signum)
    {
      case SIGINT:
        running = False;
        XNoOp(display);
        break;
      case SIGSEGV:
          {
#ifdef HAVE_EXECINFO_H
            int i, frames = 0;
            void *callstack[10] = { 0 };
            char **strings = NULL;

            frames  = backtrace(callstack, 10);
            strings = backtrace_symbols(callstack, frames);

            printf("\n\nLast %d stack frames:\n", frames);

            for(i = 0; i < frames; i++)
              printf("%s\n", strings[i]);

            free(strings);
#endif /* HAVE_EXECINFO_H */

            abort();
          }
        break;
    }
} /* }}} */

/* SubvertSelect {{{ */
static Window
SubvertSelect(void)
{
  int i, format = 0, buttons = 0;
  unsigned int n;
  unsigned long nitems = 0, bytes = 0;
  unsigned char *data = NULL;
  XEvent event;
  Window win = None;
  Atom type = None, rtype = None;
  Window dummy = None, root = None, *wins = NULL;
  Cursor cursor = None;

  root   = DefaultRootWindow(display);
  cursor = XCreateFontCursor(display, XC_cross);
  type   = XInternAtom(display, "WM_STATE", True);

  /* Grab pointer */
  if(XGrabPointer(display, root, False, ButtonPressMask|ButtonReleaseMask,
      GrabModeSync, GrabModeAsync, root, cursor, CurrentTime))
    {
      XFreeCursor(display, cursor);

      return None;
    }

  /* Select a window */
  while(None == win || 0 != buttons)
    {
      XAllowEvents(display, SyncPointer, CurrentTime);
      XWindowEvent(display, root, ButtonPressMask|ButtonReleaseMask, &event);

      switch(event.type)
        {
          case ButtonPress:
            if(None == win)
              win = event.xbutton.subwindow ? event.xbutton.subwindow : root; ///< Sanitize
            buttons++;
            break;
          case ButtonRelease:
            if(0 < buttons) buttons--;
            break;
        }
      }

  /* Find children with WM_STATE atom */
  XQueryTree(display, win, &dummy, &dummy, &wins, &n);

  for(i = 0; i < n; i++)
    {
      if(Success == XGetWindowProperty(display, wins[i], type, 0, 0, False,
          AnyPropertyType, &rtype, &format, &nitems, &bytes, &data))
        {
          if(data)
            {
              XFree(data);
              data = NULL;
            }

          if(type == rtype)
            {
              win = wins[i];

              break;
            }
        }
    }

  if(wins) XFree(wins);
  XFreeCursor(display, cursor);
  XUngrabPointer(display, CurrentTime);

  XSync(display, False); ///< Sync all changes

  return win;
} /* }}} */

/* main {{{ */
int
main(int argc,
  char *argv[])
{
  int c = 0, format = 0;
  struct sigaction sa;
  char *wmname = NULL;
  long supplied = 0;
  unsigned long bytes = 0, ndata = 0;
  unsigned char *data = NULL;
  XSizeHints shints;
  XColor xcolor = { 0 };
  XTextProperty text;
  XClassHint chint;
  Atom xa_wm_name = None, xa_wm_state = None, rtype = None;

  /* Signal handler */
  sa.sa_handler = SubvertSignal;
  sa.sa_flags   = 0;
  memset(&sa.sa_mask, 0, sizeof(sigset_t)); ///< Avoid uninitialized values
  sigemptyset(&sa.sa_mask);

  sigaction(SIGINT,  &sa, NULL);
  sigaction(SIGSEGV, &sa, NULL);

  /* Connect to display and setup error handler */
  if(!(display = XOpenDisplay(NULL)))
    {
      printf("Failed opening display\n");

      exit(-1);
    }

  XSetErrorHandler(SubvertXError);

  /* Get atoms */
  xa_wm_name  = XInternAtom(display, "WM_NAME", False);
  xa_wm_state = XInternAtom(display, "_NET_WM_STATE", False);

  printf(">>> Select a target\n");

  /* Select a window to subvert */
  if((victim = SubvertSelect()))
    {
      XWindowAttributes attrs;

      printf(">>> Hijacking #%lx\n", victim);

      /* Get attributes */
      XGetWindowAttributes(display, victim, &attrs);

      /* Create color */
      XParseColor(display, DefaultColormap(display,
        DefaultScreen(display)), "#ff0000", &xcolor);
      XAllocColor(display, DefaultColormap(display,
        DefaultScreen(display)), &xcolor);

      /* Create our window */
      culprit = XCreateSimpleWindow(display, attrs.root, attrs.x, attrs.y,
        attrs.width, attrs.height, attrs.border_width, 0, xcolor.pixel);

      /* Copy WM_NAME */
      XFetchName(display, victim, &wmname);
      XStoreName(display, culprit, wmname);

      /* Copy WM_CLASS */
      XGetClassHint(display, victim, &chint);
      XSetClassHint(display, culprit, &chint);

      printf(">>> Taking control of #%lx: name=%s, instance=%s, class=%s, "
        "x=%3d, y=%3d, w=%3d, h=%3d\n",
        victim, wmname, chint.res_name, chint.res_class, attrs.x, attrs.y,
        attrs.width, attrs.height);

      if(wmname) XFree(wmname);
      if(chint.res_class) XFree(chint.res_class);

      /* Copy size hints */
      XGetWMNormalHints(display, victim, &shints, &supplied);
      XSetWMNormalHints(display, culprit, &shints);

      /* Copy _NET_WM_STATE */
      if(Success == XGetWindowProperty(display, victim, xa_wm_state, 0L, 4096,
          False, XA_ATOM, &rtype, &format, &ndata, &bytes, &data))
        {
          XChangeProperty(display, culprit, xa_wm_state, XA_ATOM, 32,
            PropModeReplace, data, ndata);

          XFree(data);
        }

      /* Reparent and update */
      XUnmapWindow(display, victim);
      XReparentWindow(display, victim, culprit, 2, 2);
      XResizeWindow(display, victim, attrs.width - 4, attrs.height - 4);
      XSetWindowBorderWidth(display, victim, 0);
      XSelectInput(display, victim, SubstructureNotifyMask|StructureNotifyMask|
        PropertyNotify|SubstructureRedirectMask);
      XSelectInput(display, culprit, SubstructureNotifyMask|StructureNotifyMask);
      XMapWindow(display, culprit);
      XMapWindow(display, victim);

      /* Delegate and handle events */
      while(running)
        {
          XEvent ev;

          XNextEvent(display, &ev);
          switch(ev.type)
            {
              case ConfigureRequest: break;
              case ConfigureNotify:
                if(ev.xconfigure.window == culprit)
                  {
                    printf(">>> Handling configure notify (win=%#lx, x=%3d, y=%3d, "
                      "w=%3d, h=%3d)\n",
                      ev.xconfigure.window, ev.xconfigure.x, ev.xconfigure.y,
                      ev.xconfigure.width, ev.xconfigure.height);

                    XResizeWindow(display, victim,
                      ev.xconfigure.width - 4, ev.xconfigure.height - 4);

                    attrs.width  = ev.xconfigure.width;
                    attrs.height = ev.xconfigure.height;
                  }
                else if(ev.xconfigure.window == victim)
                  {
                    XMoveResizeWindow(display, victim, 2, 2,
                      attrs.width - 4, attrs.height - 4);
                  }
                break;
              case DestroyNotify:
                if(ev.xdestroywindow.window == culprit ||
                    ev.xdestroywindow.window == victim)
                  {
                    printf(">>> Handling destroy event (win=%#lx)\n",
                      ev.xdestroywindow.window);
                    running = False;

                    if(ev.xdestroywindow.window == victim) victim = None;
                  }
                break;
              case PropertyNotify:
                if(XA_WM_NAME == ev.xproperty.atom ||
                    ev.xproperty.atom == xa_wm_name)
                  {
                    XFetchName(display, victim, &wmname);
                    XStoreName(display, culprit, wmname);

                    printf(">>> Handling property notify (wmname=%s)\n",
                      wmname);

                    XFree(wmname);
                  }
              break;
            }
        }

      if(None != victim)
        {
          printf(">>> Releasing %#lx\n", victim);

          XReparentWindow(display, victim, DefaultRootWindow(display), 0, 0);
          XSetWindowBorderWidth(display, victim, attrs.border_width);
        }
      else XDestroyWindow(display, culprit);
    }

  XCloseDisplay(display);

  return 0;
}
